﻿using System.ComponentModel.DataAnnotations;

using HIPS.Web.UI.ViewModels.Shared;

namespace HIPS.Web.UI.ViewModels.PcehrView
{

    /// <summary>
    /// Base view model to be used with the PCEHR View feature.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
    /// </history>
    public class PcehrViewModelBase
    {

        #region Properties

        /// <summary>
        /// Gets or sets the list of messages associated with the view model.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public ViewMessageList Messages { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected hospital.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        [Required]
        [Display(Name = "Hospital")]
        public string HospitalId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected hospital.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        public HospitalViewModel CurrentHospital { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected patient.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public string PatientId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected patient.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="07 January 2014">Initial version.</change>
        /// </history>
        public PatientViewModel CurrentPatient { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected document repository.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public string RepositoryId { get; set; }

        /// <summary>
        /// Gets or sets the identifier associated with the selected document.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public string DocumentId { get; set; }

        /// <summary>
        /// Gets or sets the currently selected document.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="10 January 2014">Initial version.</change>
        /// </history>
        public DocumentViewModel CurrentDocument { get; set; }

        #endregion

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public PcehrViewModelBase()
        {
            this.Messages = new ViewMessageList();
        }

        #endregion

        #region Methods

        /// <summary>
        /// Determines if the view model contains a selected hospital.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected hospital, false otherwise.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public bool IsHospitalSelected()
        {
            return !(string.IsNullOrEmpty(this.HospitalId));
        }

        /// <summary>
        /// Determines if the view model contains a selected patient.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected patient, false otherwise.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public bool IsPatientSelected()
        {
            return this.IsHospitalSelected() && (!string.IsNullOrEmpty(this.PatientId));
        }

        /// <summary>
        /// Determines if the view model contains a selected document.
        /// </summary>
        /// <returns>Boolean. true if the view model contains a selected document, false otherwise.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="22 November 2013">Initial version.</change>
        /// </history>
        public bool IsDocumentSelected()
        {
            return this.IsHospitalSelected() && this.IsPatientSelected() && (!string.IsNullOrEmpty(this.RepositoryId) && !string.IsNullOrEmpty(this.DocumentId));
        }

        #endregion

    }
}